/**
* \file: WaylandConnection.h
*
* \version: 0.4
*
* \release: $Name:$
*
* Wayland utilization for SPI related use cases
*
* \component: Unified SPI
*
* \author: P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef USPI_WAYLAND_CONNECTION
#define USPI_WAYLAND_CONNECTION

#include <atomic>
#include <list>
#include <memory>
#include <pthread_adit.h>
#include <wayland-client-protocol.h>
#include "WaylandSeat.h"
#include "WaylandTouchListener.h"
#include "WaylandPointerListener.h"

namespace adit { namespace uspi {

/** @class  WaylandConnection
 *  @brief  Class to provide Wayland connection, initialization, and own the event listeners as well as input thread.
 */
class WaylandConnection
{
public:
    WaylandConnection(ITouchFacadeCallbacks& inCallbacks, int inDisplayWidth, int inDisplayHeight, int inMaxFingerNum);
    ~WaylandConnection();

    bool start(wl_display* inWlDisplay);
    bool stop();

    void setLateResolution(int inWidth, int inHeight);

private:
    ITouchFacadeCallbacks& mCallbacks;

    int mDisplayWidth, mDisplayHeight;

    wl_display* mWlDisplay;
    wl_event_queue* mInputQueue;

    /* Wayland event listeners and their objects*/
    static wl_registry_listener registryListener;
    wl_registry* mWlRegistry;

    static wl_seat_listener seatListener;
    std::list<WaylandSeat*> mSeatList;
    pthread_mutex_t mSeatListMutex;

    wl_touch_listener* mWlTouchListener;
    std::unique_ptr<WaylandTouchListener> mTouchListenerObj;

    wl_pointer_listener* mWlPointerListener;
    std::unique_ptr<WaylandPointerListener> mPointerListenerObj;

    std::atomic<bool> mInputThreadRunning;
    pthread_t mInputThread;

    int32_t shutDownEventFd;
    const uint64_t shutDownEvent = 1;

    void addWlSeat(WaylandSeat* inSeat);

    bool startInputThread();
    bool stopInputThread();
    static void* inputThread(void* inMe);

    static void onRegistryGlobal(void* inMe, wl_registry* inRegistry, uint32_t inName, const char* inInterface,
            uint32_t inVersion);
    static void onRegistryGlobalRemove(void* inMe, wl_registry* inRegistry, uint32_t inName);
    static void onSeatCapabilities(void* inData, wl_seat* inSeat, uint32_t inCaps);
    static void onSeatName(void* inData, wl_seat* inSeat, const char *name);
};

} } /* namespace adit { namespace uspi { */

#endif /* USPI_WAYLAND_CONNECTION */
